/* $Id: sca.c,v 1.10 1998/12/04 16:55:39 dhiller Exp $ */
/* Copyright (C) 1996 - 1998, Hewlett-Packard Company, all rights reserved. */

/* Functions for "customized" SCAs */

#include "sema.h"

#define CHECK(x)			\
    {					\
	rtn_val = (x);			\
	if ( rtn_val < 0 ) return rtn_val;	\
    }
#define ERROR_OUT(e)			\
    {					\
	rtn_val = (e);			\
        return i1432_print_error(rtn_val);	\
    }


static SHORTSIZ16
i1432_sca_dsp_write_prog(E1432_MODULE_LIST_NODE *mn, LONGSIZ32 length, 
  LONGSIZ32 *ldata)
{
    SHORTSIZ16 rtn_val = 0;

    if ( ldata == NULL )
    {
	/* ldata = NULL also flags no download, flag for std download instead */
	length = 0;
    }

    length >>= 2; /* convert from bytes to long words */
    if ( length > E1432_SCA_DBUF_SIZE )
    {
	ERROR_OUT(ERR1432_BUFFER_TOO_SMALL);
    }

    /* set up buffer pointers */
    CHECK(i1432_direct_write32_register(mn, E1432_SCA_BUF1_SIZE, length));
    if ( length > 0 )   /* don't need to of not downloading */
    {
        /* divide by 4 for 96k pointer */
        CHECK(i1432_direct_write32_register(mn, E1432_SCA_BUF1_PTR,
                                        E1432_SCA_DBUF >> 2));

        /* write the data */
        CHECK(i1432_xfer_writedata(mn, ldata, E1432_SCA_DBUF, length));
    }

    return rtn_val;
}

 
SHORTSIZ16
i1432_sca_dsp_download_mod(E1432ID hw, SHORTSIZ16 ID, LONGSIZ32 length,
  LONGSIZ32 *ldata)
{
    SHORTSIZ16 rtn_val = 0;
    E1432_MODULE_LIST_NODE *mn;

    /* Check for valid id */
    CHECK(i1432_checkID(hw, ID));

    /* Check if channel or group */
    if (ID < 0)
    {
	E1432_GROUP_LIST_NODE *gn = i1432_get_group_node(hw, ID);
	E1432_CHAN_LIST_NODE *cn = gn->chanlist;
	E1432_MODULE_LIST_NODE *prev_mn = NULL;

	/* iterate thru group */
	while ( cn )
	{
            CHECK(i1432_get_module_from_chan(hw, cn->chanID, &mn));
	    /* do once for each new module */
	    if ( mn != prev_mn )
	    {
	        CHECK(i1432_sca_dsp_download_mod(hw, cn->chanID, length,
		  ldata));
		prev_mn = mn;
	    }
	    cn = cn->next;
	}
    }
    else
    {
        CHECK(i1432_get_module_from_chan(hw, ID, &mn));
        CHECK(i1432_sca_dsp_write_prog(mn, length, ldata));
        CHECK(i1432_write_cmd0(hw, ID, E1432_CMD_DSP_DOWNLOAD));
    }

    return rtn_val;
}

 
SHORTSIZ16
i1432_sca_dsp_download_chan(E1432ID hw, SHORTSIZ16 ID, LONGSIZ32 length,
  LONGSIZ32 *ldata)
{
    E1432_MODULE_LIST_NODE *mn;
    E1432_GROUP_LIST_NODE *gn;
    E1432_CHAN_LIST_NODE *cn;
    SHORTSIZ16 chanID = ID;
    int cindex;
    LONGSIZ32 sca_id;
    SHORTSIZ16 rtn_val = 0;
    SHORTSIZ16 la, prev_la = -1;
    int sca, prev_sca = -1;
    int done = 0;

    /* Check for valid id */
    CHECK(i1432_checkID(hw, ID));

    /* Check if channel or group */
    if (ID < 0)
    {
	gn = i1432_get_group_node(hw, ID);
	cn = gn->chanlist;
        chanID = cn->chanID;
    }

    do
    {
        /* must be a channel that has a module allocated */
        CHECK(i1432_get_module_from_chan(hw, chanID, &mn));
        cindex = i1432_get_chan_index(hw, chanID);
	la = mn->la;

	/* for each new module, set up the download data */
	if ( la != prev_la )
	{
            CHECK(i1432_sca_dsp_write_prog(mn, length, ldata));
        }

	/* find out sca number and sca type */
	sca = i1432_get_chan_sca(hw, chanID);
	if ( la != prev_la )
	{
	    CHECK(i1432_direct_read32_register(mn,
	      E1432_SINFO_ID_REG + cindex * E1432_SINFO_REG_OFFSET, &sca_id));
        }

        /* have the module perform the download on that channel */
	if ( ! ((sca_id == E1432_SCA_ID_SONATA_A
	  || sca_id == E1432_SCA_ID_SONATA_B)
	  && la == prev_la && sca == prev_sca) )
	/* on Sonata, do once only per sca */
	{
            CHECK(i1432_write_cmd1(hw, chanID, E1432_CCMD_DSP_DOWNLOAD,
	      cindex));
	}

        if (ID < 0)
	{
	    cn = cn->next;
	    if ( cn ) chanID = cn->chanID;
	    else done = 1;
	}
	else done = 1;

        prev_la = la;
        prev_sca = sca;
    } while ( ! done );

    return rtn_val;
}

 
SHORTSIZ16 EXPORT
e1432_sca_dsp_download(E1432ID hw, SHORTSIZ16 ID, LONGSIZ32 length,
  LONGSIZ32 *ldata)
{
    return i1432_sca_dsp_download_chan(hw, ID, length, ldata);
}


SHORTSIZ16 EXPORT
e1432_dsp_exec_query(E1432ID hw, SHORTSIZ16 ID, LONGSIZ32 exec_cmd,
  LONGSIZ32 exec_data_size, LONGSIZ32 *exec_data,
  LONGSIZ32 query_data_size, LONGSIZ32 *query_data)
{
    SHORTSIZ16 rtn_val = 0;
    const LONGSIZ32 buf_start = E1432_SCA_BUF;       /* the buffer area */
    const LONGSIZ32 buf_length = E1432_SCA_BUF_SIZE; /* in long words */
    if ( exec_data_size + query_data_size > buf_length )
    {
        ERROR_OUT(ERR1432_BUFFER_TOO_SMALL);
    }

    /* Check for valid id */
    CHECK(i1432_checkID(hw, ID));

    /* Check if channel or group */
    if (ID < 0)
    {
        E1432_GROUP_LIST_NODE *gn;
        E1432_CHAN_LIST_NODE *cn;
	/* Iterate thru group */
	gn = i1432_get_group_node(hw, ID);
	cn = gn->chanlist;
	while ( cn )
	{
            CHECK(e1432_dsp_exec_query(hw, cn->chanID, exec_cmd, exec_data_size,
	      exec_data, query_data_size, query_data));
            query_data += query_data_size;
	    cn = cn->next;
	}
    }
    else
    {
        E1432_MODULE_LIST_NODE *mn;
        int i, cindex;
        LONGSIZ32 exec_buf_start, query_buf_start;
	LONGSIZ32 buf_ptr = buf_start;

        /* must be a channel that has a module allocated */
        CHECK(i1432_get_module_from_chan(hw, ID, &mn));
        cindex = i1432_get_chan_index(hw, ID);

        /* set up buffer pointers */
	exec_buf_start = (buf_start >> 2); /* divide by 4 for 96k pointer */
	/* query data buffer starts immediately after exec data buffer */
	query_buf_start = exec_buf_start + exec_data_size;
        CHECK(i1432_direct_write32_register(mn, E1432_SCA_BUF1_SIZE,
          exec_data_size));
        CHECK(i1432_direct_write32_register(mn, E1432_SCA_BUF1_PTR,
          exec_buf_start));
        CHECK(i1432_direct_write32_register(mn, E1432_SCA_BUF2_SIZE,
          query_data_size));
        CHECK(i1432_direct_write32_register(mn, E1432_SCA_BUF2_PTR,
          query_buf_start));

	/* write out the command data */
	if ( exec_data != NULL )
	{
	    for ( i = 0; i < exec_data_size; i++ )
	    {
                CHECK(i1432_direct_write32_register(mn, buf_ptr, *exec_data++));
	        buf_ptr += 4; /* 4 bytes (vxi) per word (96k) */
	    }
	}

	/* execute */
        CHECK(i1432_write_cmd2(hw, ID, E1432_CMD_DSP_EXEC_QUERY,
	  cindex, exec_cmd));
	
	/* beam up the command response */
	if ( query_data != NULL )
	{
	    for ( i = 0; i <query_data_size; i++ )
	    {
		CHECK(i1432_direct_read32_register_a(mn, buf_ptr, E1432_A24,
		  query_data++));
	        buf_ptr += 4; /* 4 bytes (vxi) per word (96k) */
	    }
	}
    }

    return rtn_val;
}
